#!/bin/sh

set -euo pipefail

# Environment variables
MYSQL_ROOT_PASSWORD=${MYSQL_ROOT_PASSWORD:-"password"}
MYSQL_WWW_DATA_PASSWORD=${MYSQL_WWW_DATA_PASSWORD:-"secure_www_data_pass"}
MYSQL_DATA_DIR="/var/lib/mysql"
MYSQL_RUN_DIR="/run/mysqld"
SQL_DUMP_PATH="/docker-entrypoint-initdb.d/db.sql"

echo "...................... [i] Starting initialization process"

# Ensure directories exist and have correct permissions
mkdir -p "$MYSQL_DATA_DIR" "$MYSQL_RUN_DIR"
chown www-data:www-data -R "$MYSQL_DATA_DIR" "$MYSQL_RUN_DIR"
chmod 700 "$MYSQL_DATA_DIR" "$MYSQL_RUN_DIR"

# Initialize database if not already present
if [ ! -d "$MYSQL_DATA_DIR/mysql" ]; then
  echo "...................... [i] MySQL data directory not found, creating initial databases"
  mariadb-install-db --user=www-data --ldata="$MYSQL_DATA_DIR"
  
  echo "...................... [i] Setting up initial privileges"
  tfile=$(mktemp)
  cat << EOF > "$tfile"
FLUSH PRIVILEGES;
GRANT ALL PRIVILEGES ON *.* TO 'root'@'localhost' WITH GRANT OPTION;
ALTER USER 'root'@'localhost' IDENTIFIED BY '$MYSQL_ROOT_PASSWORD';
GRANT ALL PRIVILEGES ON *.* TO 'www-data'@'localhost' WITH GRANT OPTION;
ALTER USER 'www-data'@'localhost' IDENTIFIED BY '$MYSQL_WWW_DATA_PASSWORD';
EOF
  /usr/bin/mariadbd --user=www-data --bootstrap < "$tfile"
  rm -f "$tfile"
else
  echo "...................... [i] MySQL data directory already initialized"
fi

echo "...................... [i] Starting MariaDB server"
/usr/bin/mariadbd --user=www-data --console --skip-name-resolve --skip-networking=0 &
sleep 5

# Import SQL dump if available
if [ -f "$SQL_DUMP_PATH" ]; then
  echo "...................... [i] Importing SQL dump from $SQL_DUMP_PATH"
  mariadb -u root -p"$MYSQL_ROOT_PASSWORD" < "$SQL_DUMP_PATH" && {
    echo "...................... [i] SQL dump imported successfully"
  } || {
    echo "...................... [w] Failed to import SQL dump, continuing anyway"
  }
else
  echo "...................... [w] No SQL dump found at $SQL_DUMP_PATH"
fi

# Start PHP-FPM
echo "...................... [i] Starting PHP-FPM"
php-fpm -D || { echo "...................... [e] Failed to start PHP-FPM"; exit 1; }

# Start Nginx
echo "...................... [i] Starting Nginx"
nginx || { echo "...................... [e] Failed to start Nginx"; exit 1; }

echo "...................... [i] Container setup complete, keeping alive"
tail -f /dev/null